const bcryptjs = require("bcryptjs");
const jwt = require("jsonwebtoken");
const { v4 } = require("uuid");
const nodemailer = require("nodemailer");
const { errorHandler } = require("../utils/error.js");
const User = require("../models/users.js");
require('dotenv').config(); 
const { sendWelcomeEmail, forgotPasswordMail } = require("../utils/notification.js");
const crypto = require('crypto');
const { imageUploadUtil } = require("../helper/cloudinary.js");
const createTransporter = require("../helper/mailer.js");

// controllers/userController.js
exports.validateEmail = async (req, res, next) => {
  try {
    console.log('validateEmail')
    const { email } = req.body;
    console.log('email', email)

    if (!email) {
      return res.status(400).json({ message: "Email is required." });
    }

    // Check if the user already exists in the database
    const existingUser = await User.findOne({ email: email.toLowerCase() });
    if (existingUser) {
      return res.status(400).json({ message: "Email already exists. Kindly login." });
    }

    // If email doesn't exist, return a success response
    console.log("Valid email")
    return res.status(200).json({ message: "Valid email." });

  } catch (error) {
    console.error(error);
    next(error);
  }
};

exports.signup = async (req, res, next) => {
  try {
      const { 
          firstName, 
          lastName, 
          email, 
          phone, 
          country, 
          address, 
          city, 
          state, 
          postalCode, 
          dateOfBirth,  
          password, 
          idType, 
          idNumber, 
          admin
      } = req.body;
      console.log('data', req.body)

      // Validate email field
      if (!email) {
          return res.status(400).json({ message: "Email is required." });
      }

      // Check if the user already exists
      const existingUser = await User.findOne({ email: email.toLowerCase() });
      if (existingUser) {
          return res.status(400).json({ message: "Email already exists. Kindly login." });
      }

      // Hash the password and generate an OTP
      const hashedPassword = bcryptjs.hashSync(password, 10);
      const otp = crypto.randomInt(1000, 9999).toString();
      const loginCode = crypto.randomInt(1000, 9999).toString();

      // Upload images to Cloudinary if they exist
      let idFileUrl = "";
      let verificationPhotoUrl = "";

      if (req.files && req.files.idFile) {
        const result = await imageUploadUtil(req.files.idFile[0].buffer);
        idFileUrl = result.secure_url;  // Now should work properly
    }
    
    if (req.files && req.files.verificationPhoto) {
        const result = await imageUploadUtil(req.files.verificationPhoto[0].buffer);
        verificationPhotoUrl = result.secure_url;  // Now should work properly
    }

      // Prepare new user data including all form fields
      const newUserData = {
          firstName,
          lastName,
          email: email.toLowerCase(),
          phone,
          country,
          address,
          city,
          state,
          postalCode,
          dateOfBirth,
          otp,
          loginCode,
          password: hashedPassword,
          idType,
          idNumber,
          idFile: idFileUrl,
          verificationPhoto: verificationPhotoUrl,
          admin,
      };

      // Create the new user
      const newUser = new User(newUserData);
      await newUser.save();

      // Send a welcome email
      await sendWelcomeEmail(newUser);

      return res.status(200).json({
          message: "Account created successfully.",
          newUser,
      });
  } catch (error) {
      console.error("Error creating user:", error);
      next(error);
  }
};

exports.signin = async (req, res, next) => {
  const { email, password } = req.body;

  try {
    const normalizedEmail = email.toLowerCase();
    const validUser = await User.findOne({ email: normalizedEmail });

    if (!validUser) return next(errorHandler(404, "User not found"));

    const validPassword = bcryptjs.compareSync(password, validUser.password);
    if (!validPassword) return next(errorHandler(401, "Wrong credentials"));

    if (!process.env.JWT_SECRET) {
      return next(errorHandler(500, "JWT secret is not defined"));
    }

    const token = jwt.sign(
      { id: validUser._id },
      process.env.JWT_SECRET,
      { expiresIn: "7d" } // good practice
    );

    const { password: pass, ...userData } = validUser.toObject();

    res.status(200).json({
      success: true,
      token,          // ✅ SEND TOKEN
      user: userData, // ✅ SEND USER
    });

  } catch (error) {
    next(error);
  }
};


exports.signOut = async (req, res, next) => {
  // console.log("signout here")
  try {
    res.clearCookie("access_token");
    res.status(200).json({
      success: true,
      message: "User has been logged out successfully!",
    });
  } catch (error) {
    next(error);
  }
};

exports.updatePassword = async (req, res, next) => {
  // Ensure the user is updating their own account
  if (req.user.id !== req.params.id) {
    return next(errorHandler(401, "You can only update your own account!"));
  }

  try {
    const { OldPassword, NewPassword } = req.body;

    // Find the user by ID
    const user = await User.findById(req.params.id);
    if (!user) {
      return next(errorHandler(404, "User not found"));
    }
     
    // Check if the old transaction pin matches
    const isPasswordMatch = await bcryptjs.compare(OldPassword, user.password);
    if (!isPasswordMatch) {
      return next(errorHandler(400, "Incorrect old password"));
    }

    // Update with the new transaction pin
    user.password = bcryptjs.hashSync(NewPassword, 10);
    await user.save();

    res.status(200).json({ message: "Password updated successfully" });
  } catch (error) {
    next(error);
  }
};


exports.forgotPassword = async (req, res, next) => {
  try {
    const { email } = req.body;
    const normalizedEmail = email.toLowerCase();
    const user = await User.findOne({ email: normalizedEmail });
    
    if (!user) {
      return res.status(404).json({ success: false, message: "User with this email does not exist." });
    }

    await forgotPasswordMail(user);

    return res.status(200).json({ success: true, message: "Password reset link sent" });
  } catch (error) {
    console.error(error); // Log the error for debugging
    return res.status(500).json({
      success: false,
      message: error.message || "Internal Server Error",
    });
  }
};

exports.resetPassword = async (req, res, next) => {
  try {
    const { otp, newPassword } = req.body;

    // Find the user by OTP
    const user = await User.findOne({ otp });

    if (!user) {
      return res.status(400).json({ message: "Invalid or expired OTP" });
    }

    // Generate a new OTP
    const newOTP = crypto.randomInt(1000, 9999).toString();

    // Hash the new password
    const hashedPassword = bcryptjs.hashSync(newPassword, 10);

    // Update the user's password and OTP
    user.password = hashedPassword;
    user.otp = newOTP;
    await user.save();

    return res.status(200).json({ message: "Password reset successfully" });
  } catch (error) {
    next(error);
  }
};


// ADMIN

exports.getAllUsers = async (req, res, next) => {
  // console.log('admin get users')
  try {
    const users = await User.find();

    if (!users || users.length === 0) {
      return res.status(404).json({ message: "No users found" });
    }

    res.status(200).json({ users });
  } catch (error) {
    console.error("Error fetching users:", error);
    res.status(500).json({ message: "Failed to fetch users" });
  }
};

exports.updateUserByAdmin = async (req, res, next) => {
  try {
    console.log("Admin updating user record");

    const {
      firstName,
      lastName,
      email,
      phone,
      country,
      address,
      city,
      state,
      postalCode,
      dateOfBirth,
      idType,
      idNumber,
      kycStatus,
      showKYC_code,
    } = req.body;

    // Build updates object safely (no undefined overwrite)
    const updates = {};

    if (firstName !== undefined) updates.firstName = firstName;
    if (lastName !== undefined) updates.lastName = lastName;
    if (email !== undefined) updates.email = email;
    if (phone !== undefined) updates.phone = phone;
    if (country !== undefined) updates.country = country;
    if (address !== undefined) updates.address = address;
    if (city !== undefined) updates.city = city;
    if (state !== undefined) updates.state = state;
    if (postalCode !== undefined) updates.postalCode = postalCode;
    if (dateOfBirth !== undefined) updates.dateOfBirth = dateOfBirth;
    if (idType !== undefined) updates.idType = idType;
    if (idNumber !== undefined) updates.idNumber = idNumber;
    if (kycStatus !== undefined) updates.kycStatus = kycStatus;
    if (showKYC_code !== undefined) updates.showKYC_code = showKYC_code;

    // Fetch current user (for status comparison or audit later)
    const currentUser = await User.findById(req.params.id);
    if (!currentUser) {
      return next(errorHandler(404, "User not found"));
    }

    // Optional: log KYC status changes
    if (
      kycStatus &&
      currentUser.kycStatus !== kycStatus
    ) {
      console.log(
        `KYC status changed from ${currentUser.kycStatus} → ${kycStatus}`
      );

      // OPTIONAL: trigger email here later
      // await sendKycStatusNotification(currentUser, kycStatus);
    }

    // Update user
    const updatedUser = await User.findByIdAndUpdate(
      req.params.id,
      { $set: updates },
      { new: true, runValidators: true }
    );

    res.status(200).json(updatedUser);
  } catch (error) {
    next(error);
  }
};

exports.getUserById = async (req, res, next) => {
  const userId = req.params.id;

  try {
    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    const { password: pass, ...rest } = user.toObject();

    res.status(200).json(rest);
  } catch (error) {
    console.error("Error fetching user:", error);
    res.status(500).json({ message: "Failed to fetch user" });
  }
};

exports.deleteUser = async (req, res, next) => {
  const userId = req.params.id;

  try {
    // Use findByIdAndDelete to delete the user directly
    const user = await User.findByIdAndDelete(userId);

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    res.status(200).json({ message: "User deleted successfully" });
  } catch (error) {
    console.error("Error deleting user:", error);
    res.status(500).json({ message: "Failed to delete user" });
  }
};


// Notification
exports.contactNotification = async (req, res) => {
  const { fullName, email, organisation, subject, message } = req.body;

  // Validation
  if (!fullName || !email || !subject || !message) {
    return res.status(400).json({
      error: "Please fill out all required fields.",
    });
  }

  try {
    const transporter = createTransporter();

    const mailOptions = {
      from: process.env.MAIL_FROM,
      to: process.env.MAIL_TO,
      replyTo: email,
      subject: `New Contact Request – ${subject}`,
      html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; background: #ffffff; padding: 24px; border-radius: 8px; border: 1px solid #e5e7eb;">
          
          <h2 style="color: #0f766e; margin-bottom: 10px;">
            New Contact Form Submission
          </h2>

          <p style="font-size: 14px; color: #475569;">
            A new enquiry has been submitted via the Anchorage Trustee Services website.
          </p>

          <hr style="margin: 20px 0;" />

          <p><strong>Full Name:</strong> ${fullName}</p>
          <p><strong>Email Address:</strong> <a href="mailto:${email}">${email}</a></p>
          ${
            organisation
              ? `<p><strong>Organisation:</strong> ${organisation}</p>`
              : ""
          }
          <p><strong>Subject:</strong> ${subject}</p>

          <p style="margin-top: 16px;"><strong>Message:</strong></p>
          <p style="white-space: pre-line; color: #334155;">
            ${message}
          </p>

          <hr style="margin: 20px 0;" />

          <p style="font-size: 12px; color: #64748b;">
            This message was submitted through the Contact page of Anchorage Trustee Services.
          </p>
        </div>
      `,
    };

    await transporter.sendMail(mailOptions);

    return res.status(200).json({
      success: true,
      message: "Your message has been sent successfully.",
    });
  } catch (error) {
    console.error("Contact email error:", error);
    return res.status(500).json({
      error: "Failed to send message. Please try again later.",
    });
  }
};




